/* XMRig
 * Copyright 2010      Jeff Garzik <jgarzik@pobox.com>
 * Copyright 2012-2014 pooler      <pooler@litecoinpool.org>
 * Copyright 2014      Lucas Jones <https://github.com/lucasjones>
 * Copyright 2014-2016 Wolf9466    <https://github.com/OhGodAPet>
 * Copyright 2016      Jay D Dee   <jayddee246@gmail.com>
 * Copyright 2016-2017 XMRig       <support@xmrig.com>
 *
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.
 */


#include <thread>


#include "amd/OclGPU.h"
#include "crypto/CryptoNight.h"
#include "Platform.h"
#include "workers/OclWorker.h"
#include "workers/OclThread.h"
#include "workers/Handle.h"
#include "workers/Workers.h"


OclWorker::OclWorker(Handle *handle) :
    m_lite(handle->isLite()),
    m_id(handle->threadId()),
    m_threads(handle->threads()),
    m_ctx(handle->ctx()),
    m_hashCount(0),
    m_timestamp(0),
    m_count(0),
    m_sequence(0)
{
    const OclThread *thread = handle->gpuThread();

    if (thread->affinity() >= 0) {
        Platform::setThreadAffinity(thread->affinity());
    }
}


void OclWorker::start()
{
    cl_uint results[0x100];

    while (Workers::sequence() > 0) {
        if (Workers::isPaused()) {
            do {
                std::this_thread::sleep_for(std::chrono::milliseconds(200));
            }
            while (Workers::isPaused());

            if (Workers::sequence() == 0) {
                break;
            }

            consumeJob();
        }

        while (!Workers::isOutdated(m_sequence)) {
            memset(results, 0, sizeof(cl_uint) * (0x100));

            XMRRunJob(m_ctx, results);

            for (size_t i = 0; i < results[0xFF]; i++) {
                *m_job.nonce() = results[i];
                Workers::submit(m_job);
            }

            m_count += m_ctx->rawIntensity;

            storeStats();
            std::this_thread::yield();
        }

        consumeJob();
    }
}


bool OclWorker::resume(const Job &job)
{
    if (m_job.poolId() == -1 && job.poolId() >= 0 && job.id() == m_pausedJob.id()) {
        m_job   = m_pausedJob;
        m_nonce = m_pausedNonce;
        return true;
    }

    return false;
}


void OclWorker::consumeJob()
{
    Job job = Workers::job();
    m_sequence = Workers::sequence();
    if (m_job == job) {
        return;
    }

    save(job);

    if (resume(job)) {
        XMRSetJob(m_ctx, m_job.blobUnsafe(), m_job.size(), m_job.target());
        return;
    }

    m_job = std::move(job);
    m_job.setThreadId(m_id);

    if (m_job.isNicehash()) {
        m_nonce = (*m_job.nonce() & 0xff000000U) + (0xffffffU / m_threads * m_id);
    }
    else {
        m_nonce = 0xffffffffU / m_threads * m_id;
    }

    m_ctx->Nonce = m_nonce;
    XMRSetJob(m_ctx, m_job.blobUnsafe(), m_job.size(), m_job.target());
}


void OclWorker::save(const Job &job)
{
    if (job.poolId() == -1 && m_job.poolId() >= 0) {
        m_pausedJob   = m_job;
        m_pausedNonce = m_nonce;
    }
}


void OclWorker::storeStats()
{
    using namespace std::chrono;

    const uint64_t timestamp = time_point_cast<milliseconds>(high_resolution_clock::now()).time_since_epoch().count();
    m_hashCount.store(m_count, std::memory_order_relaxed);
    m_timestamp.store(timestamp, std::memory_order_relaxed);
}
